<?php
/*
Objective: This API retrieves rejected requisition records from the database with pagination, including related ship and department details.
           It returns the requisition records based on the specified page and calculates the total number of pages for pagination.
File Version: 1.0
API No: KDN09
Date of Creation: 01/04/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone:

Date: 01/04/2025

Request Method:
Type: POST

Headers: Content-Type: application/json

Request Parameters:
The API expects the following parameters in the request body:
- page (int, optional) – The page number for pagination. Defaults to 1 if not provided.

Process:

* Step 1: Input Validation
  The system checks if the `page` parameter is provided in the request. If it's missing, the API defaults to page 1.

* Step 2: Establish Database Connection
  The system connects to the database using the connection class provided in the `config/config.php` file.

* Step 3: Fetch Rejected Requisition Records
  The API queries the `requisition_header_all` table to fetch rejected requisition records (status 3). The query also joins the `ship_header_all` and `department_header_all` tables to retrieve related ship and department names. Pagination is applied using the `LIMIT` and `OFFSET` clauses to fetch 6 records per page.

* Step 4: Count Total Rejected Requisitions
  The total number of rejected requisition records is calculated using `COUNT(*)` to determine how many pages of results are available.

* Step 5: Return Response
  If the requisitions are successfully fetched, the API returns a JSON response with the requisition details and total number of pages. If the query fails, an error message is returned.

Response Structure:

* Success Response:
  If the requisition records are successfully fetched, the API returns:
    - success_code: 200 (indicating success).
    - message: "Requisitions fetched successfully".
    - requests: Array of requisition details including requisition number, ship name, department name, requested date, and status.
    - totalPages: The total number of pages available based on the records and pagination limit.

* Failure Response:
  If there is an issue with the query execution or database connection, the response contains:
    - error_code: 500 (Internal Server Error).
    - message: "Failed to fetch requisition records" or "Database connection failed."

Date:

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone:

Date:
*/

session_start();
require_once("../config/config.php");

$conn = SqlServerConnection::getConnection();

$page = isset($_POST['page']) ? (int)$_POST['page'] : 1;
$limit = 6;
$offset = ($page - 1) * $limit;

$response = [];

if ($conn) {
    $sql = "SELECT r.req_id, r.req_no, r.ship_id, sha.ship_name, r.dept_id, r.close_remark, dha.dept_name, r.req_status, r.req_date_of_creation ,t.closed
            FROM requisition_header_all r 
            LEFT JOIN ship_header_all sha ON r.ship_id = sha.ship_id 
            LEFT JOIN department_header_all dha ON r.dept_id = dha.dept_id 
            INNER JOIN tracker_header_all t ON r.req_id = t.req_id
            WHERE (r.req_status = 5 OR t.closed IS NOT NULL)
            ORDER BY r.req_id DESC OFFSET ? ROWS FETCH NEXT ? ROWS ONLY"; 

    $stmt = sqlsrv_query($conn, $sql, [$offset, $limit]);

    if ($stmt) {
        $requests = [];
        while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
            // Format the date as dd/mm/yyyy
            $requested_date = $row['req_date_of_creation'] ? $row['req_date_of_creation']->format('m/d/Y') : null;

            $requests[] = [
                'requisition_no' => $row['req_no'],
                'ship_name' => $row['ship_name'],
                'department' => $row['dept_name'],
                'requested_date' => $requested_date,
                'status' => $row['req_status'],
                'close_remark' => $row['close_remark']
            ];
        }

        $sqlCount = "SELECT COUNT(*) AS total FROM requisition_header_all WHERE req_status = 5";
        $stmtCount = sqlsrv_query($conn, $sqlCount);

        if ($stmtCount) {
            $rowCount = sqlsrv_fetch_array($stmtCount, SQLSRV_FETCH_ASSOC);
            $totalRecords = $rowCount['total'];
            $totalPages = ceil($totalRecords / $limit);

            $response = [
                'success_code' => 200,
                'message' => 'Requisitions fetched successfully',
                'requests' => $requests,
                'totalPages' => $totalPages
            ];
        } else {
            $response = [
                'error_code' => 500,
                'message' => 'Failed to count requisition records: ' . print_r(sqlsrv_errors(), true)
            ];
        }

    } else {
        $response = [
            'error_code' => 500,
            'message' => 'Failed to fetch requisition records: ' . print_r(sqlsrv_errors(), true)
        ];
    }
} else {
    $response = [
        'error_code' => 500,
        'message' => 'Database connection failed.'
    ];
}

echo json_encode($response);
sqlsrv_close($conn);
?>