<?php

/*
objective : This API handles the creation and retrieval of departments in the system. 
It supports adding a new department and fetching department data with pagination and optional search functionality.

file version : 1.0
API NO : KDN02
Date of creation : 26/03/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone : 

Date  : 26/03/2025

Request Method
Type: POST, GET

Headers: Content-Type: application/json

Request Parameters
- For POST:
  - department_name (string, required) – The name of the department to be added.

- For GET:
  - search (string, optional) – Search query for filtering department names.
  - page (int, optional) – The page number for pagination. Default is 1.

Authentication: No authentication is required for this API.

***********

Authentication Process

* Step 1: Input Validation
  - For POST, it checks if the department name is provided.
  - For GET, it checks for the search and pagination parameters.
  - If required fields are missing, the API returns an error response with an appropriate message.

* Step 2: Duplicate Check (POST only)
  - The system checks if the department already exists in the database.
  - If a duplicate department is found, an error response is returned.

* Step 3: Inserting Department (POST only)
  - If no duplicate is found, a new department is inserted into the database.
  - A unique department number (dept_no) is generated using a naming convention based on the department name.

* Step 4: Retrieving Department Data (GET only)
  - For GET requests, it returns a paginated list of departments, optionally filtered by the search query.
  - It calculates the total number of pages for pagination based on the number of matching departments.

***********

Response Structure

* Success Response
  - For POST:
    - success_code: 200 (indicating success)
    - message: 'Department added successfully'

  - For GET:
    - error_code: 200 (indicating success)
    - data: Array of department details
    - total_pages: Total number of pages for pagination
    - current_page: The current page number

* Failure Responses:
  - For POST:
    - error_code: 100 (Department already exists)
    - error_code: 400 (Missing required department name)
    - error_code: 102 (Error while adding department)

  - For GET:
    - error_code: 101 (No departments found)
    - error_code: 300 (Invalid request method)

Date :
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone : 

Date : 
*/


session_start();
require_once("../config/config.php");

// Set the number of items per page
$itemsPerPage = 5;
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['department_name']) && !empty($_POST['department_name'])) {
        $conn = connection::getConnection();

        // Sanitize input to prevent SQL injection
        $dept_name = mysqli_real_escape_string($conn, $_POST['department_name']);
        $dept_status = 1; // Active status
        $inserted_on = date('Y-m-d H:i:s');
        $suspended_on = null;

        // Query to check duplicate department
        $check_sql = "SELECT dept_name FROM department_header_all WHERE dept_name = '$dept_name'";
        $check_result = mysqli_query($conn, $check_sql);

        if (mysqli_num_rows($check_result) > 0) {
            echo json_encode(['success' => false, 'message' => 'Department already exists']);
        } else {
            $dept_no = generateDeptNo($dept_name, $conn);

            // SQL query to insert department into the database
            $sql = "INSERT INTO department_header_all (dept_no, dept_name, dept_status, inserted_on, suspended_on) 
                    VALUES ('$dept_no', '$dept_name', '$dept_status', '$inserted_on', '$suspended_on')";

            if (mysqli_query($conn, $sql)) {
                echo json_encode(['success' => true, 'message' => 'Department added successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Error adding department: ' . mysqli_error($conn)]);
            }
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Department name is required']);
    }
} elseif ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $conn = connection::getConnection();
    $search = isset($_GET['search']) ? mysqli_real_escape_string($conn, $_GET['search']) : '';
    $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;

   
    // Query to get total department count for pagination
    $totalDepartmentsQuery = "SELECT COUNT(*) AS total FROM department_header_all WHERE dept_name LIKE '%$search%'";
    $totalDepartmentsResult = mysqli_query($conn, $totalDepartmentsQuery);
    $totalDepartments = mysqli_fetch_assoc($totalDepartmentsResult)['total'];

    // Calculate total pages
    $totalPages = ceil($totalDepartments / $itemsPerPage);
    $currentPage = max(1, min($currentPage, $totalPages));

    // Calculate the starting index for the current page
    $start = ($currentPage - 1) * $itemsPerPage;

    // Query to fetch departments for the current page
    if ($search) {
        $sql = "SELECT dept_no, dept_name, dept_status FROM department_header_all WHERE dept_name LIKE '%$search%' LIMIT $start, $itemsPerPage";
    } else {
        $sql = "SELECT dept_no, dept_name, dept_status FROM department_header_all LIMIT $start, $itemsPerPage";
    }

    $result = mysqli_query($conn, $sql);

    $departments = [];
    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $departments[] = $row;
        }

        // Return the departments along with pagination info
        echo json_encode([
            'success' => true,
            'data' => $departments,
            'total_pages' => $totalPages,
            'current_page' => $currentPage
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'No departments found']);
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
}

// Function to generate a unique dept_no
function generateDeptNo($dept_name, $conn) {
    $letters = preg_replace('/[^A-Za-z]/', '', $dept_name);
    $dept_prefix = strtoupper(substr($letters, 0, 2));

    // Query to get the last department number for the same department prefix
    $sql = "SELECT dept_no FROM department_header_all WHERE dept_no LIKE '$dept_prefix%' ORDER BY dept_no DESC LIMIT 1";
    $result = mysqli_query($conn, $sql);

    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $last_dept_no = $row['dept_no'];
        $last_serial = (int) substr($last_dept_no, 3);
        $new_serial = str_pad($last_serial + 1, 2, '0', STR_PAD_LEFT);
    } else {
        $new_serial = '01';
    }

    return $dept_prefix . '-' . $new_serial;
}
?>
