<?php

/*
objective : This API handles the addition of new items and retrieval of items with pagination and filtering options. It supports both POST and GET methods. 
            - POST is used for adding new items to the database.
            - GET is used for retrieving departments, categories, subcategories, or paginated items with search functionality.
file version : 1.0
API NO : KDN06
Date of creation : 28/03/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone :

Date  : 28/03/2025

Request Method
Type: POST, GET

Headers: Content-Type: application/json

Request Parameters
The API expects the following parameters in the request body for POST:
- dept_id (integer, required) – The department ID of the item.
- cat_id (integer, required) – The category ID of the item.
- sub_cat_id (integer, required) – The subcategory ID of the item.
- item_name (string, required) – The name of the item.
- drawing_no (string, required) – The drawing number of the item.
- part_no (string, required) – The part number of the item.

For GET:
- action (string, required) – Action to perform (fetch_departments, fetch_categories, fetch_subcategories).
- dept_id (integer, optional) – The department ID used to filter categories and subcategories.
- cat_id (integer, optional) – The category ID used to filter subcategories.
- search (string, optional) – A search term to filter items by name.
- page (integer, optional) – The current page number for pagination.

***********

Request Handling Process

*Step 1: POST Request
The API checks if all required fields are present in the POST request body.
- If any field is missing or empty, it returns an error response: "All fields are required."
- If the item already exists in the same department, category, and subcategory, it returns an error message: "This item already exists in the selected department, category, and subcategory."
- Otherwise, it inserts the new item into the database and returns a success message with item creation details.

*Step 2: GET Request
The API supports multiple actions:
- "fetch_departments": Fetches the list of all departments.
- "fetch_categories": Fetches categories based on the provided department ID.
- "fetch_subcategories": Fetches subcategories based on the provided category ID.
- Default action: Fetches items with pagination and optional search functionality. The items are returned along with pagination details such as total pages and current page.

*Step 3: Input Validation and Sanitization
- All inputs (parameters) are sanitized using `mysqli_real_escape_string` to prevent SQL injection attacks.
- Pagination is applied with a limit of 5 items per page.

********

Response Structure

*Success Response
- For POST request (adding items):
  - success_code: 200 (indicating success)
  - message: "Item added successfully"

- For GET request (retrieving data):
  - success: true
  - data: The retrieved data (departments, categories, subcategories, or items)
  - total_pages: Total number of pages (for pagination)
  - current_page: Current page number

*Failure Responses
- If required fields are missing in POST request:
  - error_code: 400
  - message: "All fields are required"

- If the item already exists:
  - error_code: 100
  - message: "This item already exists in the selected department, category, and subcategory."

- For GET request (no items found):
  - error_code: 101
  - message: "No items found"

- If the request method is invalid (neither POST nor GET):
  - error_code: 300
  - message: "Invalid request method"

-------------------------------------------------------------------------------------------------------------------------------

Date :

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone :

Date : 

*/

session_start();
require_once("../config/config.php");

$itemsPerPage = 5;

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['dept_id']) && isset($_POST['cat_id']) && isset($_POST['sub_cat_id']) && isset($_POST['item_name']) && isset($_POST['drawing_no']) && isset($_POST['part_no']) && !empty($_POST['dept_id']) && !empty($_POST['item_name'])) {
        $conn = connection::getConnection();

        // Sanitize input to prevent SQL injection
        $dept_id = mysqli_real_escape_string($conn, $_POST['dept_id']);
        $cat_id = mysqli_real_escape_string($conn, $_POST['cat_id']);
        $sub_cat_id = mysqli_real_escape_string($conn, $_POST['sub_cat_id']);
        $item_name = mysqli_real_escape_string($conn, $_POST['item_name']);
        $drawing_no = mysqli_real_escape_string($conn, $_POST['drawing_no']);
        $part_no = mysqli_real_escape_string($conn, $_POST['part_no']);
        $date_of_creation = date('Y-m-d H:i:s');

        // Check if item already exists in the same department, category, and subcategory
        $checkQuery = "SELECT item_name FROM item_header_all WHERE dept_id = '$dept_id' AND cat_id = '$cat_id' AND sub_cat_id = '$sub_cat_id' AND item_name = '$item_name'";
        $checkResult = mysqli_query($conn, $checkQuery);

        if (mysqli_num_rows($checkResult) > 0) {
            echo json_encode(['success' => false, 'message' => 'This item already exists in the selected department, category, and subcategory.']);
        } else {
            // Insert Item into item_header_all table
            $sql = "INSERT INTO item_header_all (dept_id, cat_id, sub_cat_id, item_name, item_drawing_no, item_part_no, item_date_of_creation) 
                    VALUES ('$dept_id', '$cat_id', '$sub_cat_id', '$item_name', '$drawing_no', '$part_no', '$date_of_creation')";

            if (mysqli_query($conn, $sql)) {
                echo json_encode(['success' => true, 'message' => 'Item added successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Error adding item: ' . mysqli_error($conn)]);
            }
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'All fields are required']);
    }
} elseif ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $conn = connection::getConnection();
    if (isset($_GET['action']) && $_GET['action'] == 'fetch_departments') {
        $sql = "SELECT dept_id, dept_name FROM department_header_all";
        $result = mysqli_query($conn, $sql);

        $departments = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $departments[] = $row;
        }
        echo json_encode($departments);
    } elseif (isset($_GET['action']) && $_GET['action'] == 'fetch_categories') {
        $dept_id = isset($_GET['dept_id']) ? (int)$_GET['dept_id'] : 0;
        $sql = "SELECT cat_id, cat_name FROM category_header_all WHERE dept_id = '$dept_id'";
        $result = mysqli_query($conn, $sql);

        $categories = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $categories[] = $row;
        }
        echo json_encode($categories);
    } elseif (isset($_GET['action']) && $_GET['action'] == 'fetch_subcategories') {
        $cat_id = isset($_GET['cat_id']) ? (int)$_GET['cat_id'] : 0;
        $sql = "SELECT sub_cat_id, sub_cat_name FROM sub_category_header_all WHERE cat_id = '$cat_id'";

        $result = mysqli_query($conn, $sql);

        $subcategories = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $subcategories[] = $row;
        }
        echo json_encode($subcategories);
    } else {
        $search = isset($_GET['search']) ? mysqli_real_escape_string($conn, $_GET['search']) : '';
        $dept_id = isset($_GET['dept_id']) ? (int)$_GET['dept_id'] : 0;
        $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;

        // Query to get total item count for pagination with search
        $totalItemsQuery = "SELECT COUNT(*) AS total FROM item_header_all WHERE item_name LIKE '%$search%'";
        if ($dept_id) {
            $totalItemsQuery .= " AND dept_id = '$dept_id'";
        }
        $totalItemsResult = mysqli_query($conn, $totalItemsQuery);
        $totalItems = mysqli_fetch_assoc($totalItemsResult)['total'];
        $totalPages = ceil($totalItems / $itemsPerPage);
        $currentPage = max(1, min($currentPage, $totalPages));
        $start = ($currentPage - 1) * $itemsPerPage;

        // Query to fetch items for the current page with search
        $sql = "SELECT i.item_id, i.item_name, i.item_drawing_no, i.item_part_no, d.dept_name, c.cat_name, s.sub_cat_name 
                FROM item_header_all i 
                LEFT JOIN department_header_all d ON i.dept_id = d.dept_id
                LEFT JOIN category_header_all c ON i.cat_id = c.cat_id
                LEFT JOIN sub_category_header_all s ON i.sub_cat_id = s.sub_cat_id
                WHERE i.item_name LIKE '%$search%'";

        if ($dept_id) {
            $sql .= " AND i.dept_id = '$dept_id'";
        }

        $sql .= " LIMIT $start, $itemsPerPage";

        $result = mysqli_query($conn, $sql);

        $items = [];
        if ($result && mysqli_num_rows($result) > 0) {
            while ($row = mysqli_fetch_assoc($result)) {
                $items[] = $row;
            }
            echo json_encode([
                'success' => true,
                'data' => $items,
                'total_pages' => $totalPages,
                'current_page' => $currentPage
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'No items found']);
        }
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
}
?>
